package mcfall.raytracer;

//TODO Write REAL DOCUMENTATION
import java.awt.image.BufferedImage;
import java.awt.image.WritableRaster;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.Observable;
import java.util.Observer;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import javax.swing.ImageIcon;
import javax.swing.JLabel;

/**
 * The Class RayTracerImageLabel.
 */
public class RayTracerImageLabel extends JLabel implements Observer {
	
	static DecimalFormat numberFormatter;
	
	static {
		//numberFormatter = NumberFormat.getNumberInstance();
		numberFormatter= new DecimalFormat ("#,###.00");		
	}
	
	/** The max threads. */
	int maxThreads; //set to the number of processors for speed, the number of subdivisions for testing (allows you to see more of the image at the same time)
	int totalThreads=0;
	int threadsFinished = 0;
	long timeStarted;
	long timeFinished = -1;
	long pixelsRendered = 0;
	Updater updater;
	/** The ray tracer. */
	RayTracer rayTracer;
	
	/** The raster. */
	WritableRaster raster;
	
	/** The image. */
	BufferedImage image;
	
	/** The executor service. */
	ExecutorService executorService;
	RayTracerImageLabel(RayTracer rayTracer, int horizontalAlignment, int maxThreads) {
		this(rayTracer, horizontalAlignment);
		this.maxThreads = maxThreads;
	}
	/**
	 * Instantiates a new ray tracer image label.
	 * 
	 * @param rayTracer the ray tracer
	 * @param horizontalAlignment the horizontal alignment
	 */
	private RayTracerImageLabel(RayTracer rayTracer, int horizontalAlignment) {
		setHorizontalAlignment(horizontalAlignment);
		this.rayTracer = rayTracer;
		rayTracer.addObserver(this);
		image = new BufferedImage (rayTracer.getImageWidth(), rayTracer.getImageHeight(), BufferedImage.TYPE_INT_ARGB);
		raster = image.getRaster();
		updateIcon();
	}
	
	
	
	/**
	 * Start trace.
	 * 
	 * @param subrow the subrow
	 * @param subcol the subcol
	 */
	public void startTrace(int subrow, int subcol) 
	{
		updater = new Updater();
		updater.start();
		timeStarted = System.currentTimeMillis();
		setExecutorService(Executors.newFixedThreadPool(maxThreads));
		int pixelsPerRowSub = (int)Math.ceil(((double)rayTracer.getImageHeight())/((double)subrow));
		int pixelsPerColSub = (int)Math.ceil(((double)rayTracer.getImageWidth())/((double)subcol));
		for(int row=0;row<rayTracer.getImageHeight();row+=pixelsPerRowSub) 
		{
			int rowheight = Math.min(pixelsPerRowSub,(rayTracer.getImageHeight()-row));
			for(int col=0;col<rayTracer.getImageWidth();col+=pixelsPerColSub) 
			{
				totalThreads++;
				int colwidth = Math.min(pixelsPerColSub,(rayTracer.getImageWidth()-col));
				WritableRaster r = raster.createWritableChild(col, row, colwidth,rowheight , col, row, null);
				executorService.execute(new RayTracerBatch(rayTracer,r, col,row,col+colwidth,row+rowheight));
			}
		}
		executorService.shutdown();
	}
	protected long getMillisElapsed() 
	{
		if(!isDone()&&timeFinished<=0) {
			return System.currentTimeMillis()-timeStarted;
		} else {			
			return timeFinished-timeStarted;
		}
	}
	public double getPixelsPerSecond()
	{
		if(getMillisElapsed()!=0) {
			return 1000d*(double)pixelsRendered/(double)getMillisElapsed();
		} else {
			return 0;
		}
	}
	protected double getPercentageDone()
	{
		if(totalThreads!=0) {
			return getPixelsPerSecond()/getTotalPixels();
		} else {
			return 0;
		}
	}
	protected double getTotalTime()
	{
		return getTotalPixels()/getPixelsPerSecond();
	}
	protected double getSecondsRemaining()
	{
		return getTotalTime()-getMillisElapsed()/1000d;
	}
	protected double getTotalPixels() {
		return (double)rayTracer.getImageHeight()*(double)rayTracer.getImageWidth();
	}
	public boolean isDone()
	{
		if(executorService!=null) {
			return executorService.isTerminated();
		} else {
			return false; //we havent even started yet
		}
	}
	
	
	
	
	/**
	 * Gets the executor service.
	 * 
	 * @return the executor service
	 */
	protected ExecutorService getExecutorService() {
		return executorService;
	}
	
	/**
	 * Sets the executor service.
	 * 
	 * @param executorService the new executor service
	 */
	protected void setExecutorService(ExecutorService executorService) {
		this.executorService = executorService;
	}
	/**
	 * Gets the max threads.
	 * 
	 * @return the max threads
	 */
	public int getMaxThreads() {
		return maxThreads;
	}
	
	/**
	 * Sets the max threads.
	 * 
	 * @param maxThreads the new max threads
	 */
	public void setMaxThreads(int maxThreads) {
		this.maxThreads = maxThreads;
	}
	
	/**
	 * Gets the image.
	 * 
	 * @return the image
	 */
	public BufferedImage getImage()
	{
		return image;
	}
	/**
	 * Update icon.
	 */
	public synchronized void updateIcon() 
	{
		ImageIcon ii = new ImageIcon(image);
		this.setIcon(ii);
		
		StringBuffer labelText = new StringBuffer (256);
		labelText.append ("Pixels per second: ");
		labelText.append (numberFormatter.format(this.getPixelsPerSecond()));
		labelText.append (" Elapsed time: ");
		labelText.append (numberFormatter.format(this.getMillisElapsed()/1000d));
		labelText.append (" sec");
		labelText.append (" Time remaining: ");
		labelText.append (Math.round(this.getSecondsRemaining()/60));
		labelText.append (" min");		 
		this.setText(labelText.toString());
		if(isDone()&&timeFinished<=0) 
		{
			if(updater!=null) { updater.kill(); }
			timeFinished = System.currentTimeMillis();
			updateIcon();
		}
	}

	public void update(Observable arg0, Object arg1) {
		if(arg0 instanceof RayTracer)	 {
			pixelsRendered=((RayTracer)arg0).getPixelsDone();
			updater.wakeUp();
		} else {
			System.out.println("A non-RayTracer object called this object...the updater will not work");
		}
	}
	/**
	 * The Class Updater.
	 */
	private class Updater extends Thread
	{
		private boolean kill = false;
		public synchronized void kill() {
			kill=true;
		}
		public synchronized boolean isKilled() {
			return kill;
		}
		public void wakeUp() {
			this.interrupt();
		}
		@Override
		public void run() {
			this.setPriority(Thread.MIN_PRIORITY);
			while(isKilled()==false) { 
				updateIcon();
				try {
					this.sleep(500);//wait for 1/2s unless otherwise notified
				} catch (InterruptedException e1) {
					//awoken
				}
			}
		}		
	}
	
}
